<?php
/**
 * AI Tools Explorer functions and definitions
 *
 * @package AI_Tools_Explorer
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Define theme constants
define( 'AI_TOOLS_EXPLORER_VERSION', '1.0.0' );
define( 'AI_TOOLS_EXPLORER_DIR', trailingslashit( get_template_directory() ) );
define( 'AI_TOOLS_EXPLORER_URI', trailingslashit( get_template_directory_uri() ) );

/**
 * Sets up theme defaults and registers support for various WordPress features.
 */
function ai_tools_explorer_setup() {
	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	// Let WordPress manage the document title.
	add_theme_support( 'title-tag' );

	// Enable support for Post Thumbnails on posts and pages.
	add_theme_support( 'post-thumbnails' );

	// Register nav menus
	register_nav_menus(
		array(
			'primary' => esc_html__( 'Primary Menu', 'ai-tools-explorer' ),
			'footer'  => esc_html__( 'Footer Menu', 'ai-tools-explorer' ),
		)
	);

	// Switch default core markup to output valid HTML5.
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
			'style',
			'script',
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );
	
	// Align wide support
	add_theme_support( 'align-wide' );
	
	// Editor styles
	add_theme_support( 'editor-styles' );
	
	// Responsive embeds
	add_theme_support( 'responsive-embeds' );
}
add_action( 'after_setup_theme', 'ai_tools_explorer_setup' );

/**
 * Enqueue scripts and styles.
 */
function ai_tools_explorer_scripts() {
	// Main stylesheet
	wp_enqueue_style( 'ai-tools-explorer-style', get_stylesheet_uri(), array(), AI_TOOLS_EXPLORER_VERSION );

	// Custom CSS
	wp_enqueue_style( 'ai-tools-explorer-custom', AI_TOOLS_EXPLORER_URI . 'src/css/styles.css', array(), AI_TOOLS_EXPLORER_VERSION );
	
	// FontAwesome
	wp_enqueue_style( 'font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css', array(), '6.5.1' );
	
	// AOS CSS
	wp_enqueue_style( 'aos', 'https://cdn.jsdelivr.net/npm/aos@2.3.4/dist/aos.css', array(), '2.3.4' );
	
	// Particles.js
	wp_enqueue_script( 'particles-js', 'https://cdn.jsdelivr.net/npm/particles.js@2.0.0/particles.min.js', array(), '2.0.0', true );
	
	// AOS JS
	wp_enqueue_script( 'aos-js', 'https://cdn.jsdelivr.net/npm/aos@2.3.4/dist/aos.js', array(), '2.3.4', true );
	
	// GSAP
	wp_enqueue_script( 'gsap', 'https://cdn.jsdelivr.net/npm/gsap@3.12.4/dist/gsap.min.js', array(), '3.12.4', true );
	
	// Main JS
	wp_enqueue_script( 'ai-tools-explorer-js', AI_TOOLS_EXPLORER_URI . 'src/js/main.js', array('jquery'), AI_TOOLS_EXPLORER_VERSION, true );
}
add_action( 'wp_enqueue_scripts', 'ai_tools_explorer_scripts' );

/**
 * Register widget area.
 */
function ai_tools_explorer_widgets_init() {
	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer 1', 'ai-tools-explorer' ),
			'id'            => 'footer-1',
			'description'   => esc_html__( 'Add widgets here.', 'ai-tools-explorer' ),
			'before_widget' => '<div class="footer-widget">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		)
	);
	
	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer 2', 'ai-tools-explorer' ),
			'id'            => 'footer-2',
			'description'   => esc_html__( 'Add widgets here.', 'ai-tools-explorer' ),
			'before_widget' => '<div class="footer-widget">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		)
	);
	
	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer 3', 'ai-tools-explorer' ),
			'id'            => 'footer-3',
			'description'   => esc_html__( 'Add widgets here.', 'ai-tools-explorer' ),
			'before_widget' => '<div class="footer-widget">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		)
	);
}
add_action( 'widgets_init', 'ai_tools_explorer_widgets_init' );

/**
 * Create custom post type for AI Tools
 */
function ai_tools_explorer_custom_post_types() {
	// AI Tools Custom Post Type
	register_post_type(
		'ai_tool',
		array(
			'labels' => array(
				'name'               => esc_html__( 'AI Tools', 'ai-tools-explorer' ),
				'singular_name'      => esc_html__( 'AI Tool', 'ai-tools-explorer' ),
				'add_new'            => esc_html__( 'Add New', 'ai-tools-explorer' ),
				'add_new_item'       => esc_html__( 'Add New AI Tool', 'ai-tools-explorer' ),
				'edit_item'          => esc_html__( 'Edit AI Tool', 'ai-tools-explorer' ),
				'new_item'           => esc_html__( 'New AI Tool', 'ai-tools-explorer' ),
				'view_item'          => esc_html__( 'View AI Tool', 'ai-tools-explorer' ),
				'search_items'       => esc_html__( 'Search AI Tools', 'ai-tools-explorer' ),
				'not_found'          => esc_html__( 'No AI tools found', 'ai-tools-explorer' ),
				'not_found_in_trash' => esc_html__( 'No AI tools found in Trash', 'ai-tools-explorer' ),
			),
			'public'              => true,
			'has_archive'         => true,
			'show_in_rest'        => true,
			'menu_icon'           => 'dashicons-superhero',
			'supports'            => array( 'title', 'editor', 'thumbnail', 'excerpt', 'custom-fields' ),
			'menu_position'       => 5,
			'rewrite'             => array( 'slug' => 'ai-tool' ),
		)
	);
	
	// Tool Categories Taxonomy
	register_taxonomy(
		'tool_category',
		'ai_tool',
		array(
			'labels' => array(
				'name'              => esc_html__( 'Tool Categories', 'ai-tools-explorer' ),
				'singular_name'     => esc_html__( 'Tool Category', 'ai-tools-explorer' ),
				'search_items'      => esc_html__( 'Search Categories', 'ai-tools-explorer' ),
				'all_items'         => esc_html__( 'All Categories', 'ai-tools-explorer' ),
				'parent_item'       => esc_html__( 'Parent Category', 'ai-tools-explorer' ),
				'parent_item_colon' => esc_html__( 'Parent Category:', 'ai-tools-explorer' ),
				'edit_item'         => esc_html__( 'Edit Category', 'ai-tools-explorer' ),
				'update_item'       => esc_html__( 'Update Category', 'ai-tools-explorer' ),
				'add_new_item'      => esc_html__( 'Add New Category', 'ai-tools-explorer' ),
				'new_item_name'     => esc_html__( 'New Category Name', 'ai-tools-explorer' ),
				'menu_name'         => esc_html__( 'Categories', 'ai-tools-explorer' ),
			),
			'hierarchical'      => true,
			'show_ui'           => true,
			'show_admin_column' => true,
			'query_var'         => true,
			'show_in_rest'      => true,
			'rewrite'           => array( 'slug' => 'tool-category' ),
		)
	);
}
add_action( 'init', 'ai_tools_explorer_custom_post_types' );

/**
 * Add meta boxes for AI Tools
 */
function ai_tools_explorer_meta_boxes() {
	add_meta_box(
		'ai_tool_details',
		esc_html__( 'AI Tool Details', 'ai-tools-explorer' ),
		'ai_tools_explorer_tool_details_callback',
		'ai_tool',
		'normal',
		'high'
	);
}
add_action( 'add_meta_boxes', 'ai_tools_explorer_meta_boxes' );

/**
 * Meta box callback
 */
function ai_tools_explorer_tool_details_callback( $post ) {
	wp_nonce_field( 'ai_tools_explorer_save_meta', 'ai_tools_explorer_meta_nonce' );
	
	$tool_url = get_post_meta( $post->ID, '_tool_url', true );
	$pricing = get_post_meta( $post->ID, '_pricing', true );
	$rating = get_post_meta( $post->ID, '_rating', true );
	$features_rating = get_post_meta( $post->ID, '_features_rating', true );
	$pricing_rating = get_post_meta( $post->ID, '_pricing_rating', true );
	$ease_of_use_rating = get_post_meta( $post->ID, '_ease_of_use_rating', true );
	$quality_rating = get_post_meta( $post->ID, '_quality_rating', true );
	
	?>
	<p>
		<label for="tool_url"><?php esc_html_e( 'Tool URL', 'ai-tools-explorer' ); ?></label><br>
		<input type="url" id="tool_url" name="tool_url" value="<?php echo esc_url( $tool_url ); ?>" class="widefat">
	</p>
	<p>
		<label for="pricing"><?php esc_html_e( 'Pricing (e.g. Free / $20 per month)', 'ai-tools-explorer' ); ?></label><br>
		<input type="text" id="pricing" name="pricing" value="<?php echo esc_attr( $pricing ); ?>" class="widefat">
	</p>
	<p>
		<label for="rating"><?php esc_html_e( 'Overall Rating (0-5)', 'ai-tools-explorer' ); ?></label><br>
		<input type="number" id="rating" name="rating" value="<?php echo esc_attr( $rating ); ?>" min="0" max="5" step="0.1" class="widefat">
	</p>
	<h4><?php esc_html_e( 'Category Ratings', 'ai-tools-explorer' ); ?></h4>
	<p>
		<label for="features_rating"><?php esc_html_e( 'Features Rating (0-5)', 'ai-tools-explorer' ); ?></label><br>
		<input type="number" id="features_rating" name="features_rating" value="<?php echo esc_attr( $features_rating ); ?>" min="0" max="5" step="0.1" class="widefat">
	</p>
	<p>
		<label for="pricing_rating"><?php esc_html_e( 'Pricing Rating (0-5)', 'ai-tools-explorer' ); ?></label><br>
		<input type="number" id="pricing_rating" name="pricing_rating" value="<?php echo esc_attr( $pricing_rating ); ?>" min="0" max="5" step="0.1" class="widefat">
	</p>
	<p>
		<label for="ease_of_use_rating"><?php esc_html_e( 'Ease of Use Rating (0-5)', 'ai-tools-explorer' ); ?></label><br>
		<input type="number" id="ease_of_use_rating" name="ease_of_use_rating" value="<?php echo esc_attr( $ease_of_use_rating ); ?>" min="0" max="5" step="0.1" class="widefat">
	</p>
	<p>
		<label for="quality_rating"><?php esc_html_e( 'Quality Rating (0-5)', 'ai-tools-explorer' ); ?></label><br>
		<input type="number" id="quality_rating" name="quality_rating" value="<?php echo esc_attr( $quality_rating ); ?>" min="0" max="5" step="0.1" class="widefat">
	</p>
	<?php
}

/**
 * Save meta box data
 */
function ai_tools_explorer_save_meta( $post_id ) {
	if ( ! isset( $_POST['ai_tools_explorer_meta_nonce'] ) ) {
		return;
	}
	
	if ( ! wp_verify_nonce( $_POST['ai_tools_explorer_meta_nonce'], 'ai_tools_explorer_save_meta' ) ) {
		return;
	}
	
	if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
		return;
	}
	
	if ( ! current_user_can( 'edit_post', $post_id ) ) {
		return;
	}
	
	if ( isset( $_POST['tool_url'] ) ) {
		update_post_meta( $post_id, '_tool_url', sanitize_url( $_POST['tool_url'] ) );
	}
	
	if ( isset( $_POST['pricing'] ) ) {
		update_post_meta( $post_id, '_pricing', sanitize_text_field( $_POST['pricing'] ) );
	}
	
	if ( isset( $_POST['rating'] ) ) {
		update_post_meta( $post_id, '_rating', floatval( $_POST['rating'] ) );
	}
	
	if ( isset( $_POST['features_rating'] ) ) {
		update_post_meta( $post_id, '_features_rating', floatval( $_POST['features_rating'] ) );
	}
	
	if ( isset( $_POST['pricing_rating'] ) ) {
		update_post_meta( $post_id, '_pricing_rating', floatval( $_POST['pricing_rating'] ) );
	}
	
	if ( isset( $_POST['ease_of_use_rating'] ) ) {
		update_post_meta( $post_id, '_ease_of_use_rating', floatval( $_POST['ease_of_use_rating'] ) );
	}
	
	if ( isset( $_POST['quality_rating'] ) ) {
		update_post_meta( $post_id, '_quality_rating', floatval( $_POST['quality_rating'] ) );
	}
}
add_action( 'save_post', 'ai_tools_explorer_save_meta' );

/**
 * Elementor support
 */
function ai_tools_explorer_elementor_support() {
	// Add support for Elementor Pro Theme Builder
	add_theme_support( 'elementor-pro' );
}
add_action( 'after_setup_theme', 'ai_tools_explorer_elementor_support' );